/* -*-pgsql-c-*- */
/*
 * $Header: /cvsroot/pgpool/pgpool/pool_process_query.c,v 1.13 2006/02/14 08:19:06 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003-2006	PgPool Global Development Group
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * pool_process_query.c: query processing stuff
 *
*/
#include "config.h"
#include <errno.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <ctype.h>

#include "pool.h"

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION_POOL *backend);

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION_POOL *backend, char *query);

static POOL_STATUS Execute(POOL_CONNECTION *frontend, 
						   POOL_CONNECTION_POOL *backend);

#ifdef NOT_USED
static POOL_STATUS Sync(POOL_CONNECTION *frontend, 
						   POOL_CONNECTION_POOL *backend);
#endif

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION_POOL *backend, int send_ready);

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION_POOL *backend);

static POOL_STATUS CopyDataRows(POOL_CONNECTION *frontend,
								POOL_CONNECTION_POOL *backend, int copyin);

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend);

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION_POOL *backend);

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend);

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION_POOL *backend,
							short num_fields);

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION_POOL *backend,
							 short num_fields);

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION_POOL *backend);

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION_POOL *backend);

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend);

static int synchronize(POOL_CONNECTION *cp);
static void process_reporting(POOL_CONNECTION *frontend, POOL_CONNECTION_POOL *backend);
static int reset_backend(POOL_CONNECTION_POOL *backend, int qcnt);

static int load_balance_enabled(POOL_CONNECTION_POOL *backend, char *sql);
static void start_load_balance(POOL_CONNECTION_POOL *backend);
static void end_load_balance(POOL_CONNECTION_POOL *backend);
static POOL_STATUS do_command(POOL_CONNECTION *backend, char *query, int protoMajor, int no_ready_for_query);
static int need_insert_lock(POOL_CONNECTION_POOL *backend, char *query);
static POOL_STATUS insert_lock(POOL_CONNECTION_POOL *backend, char *query);
static char *get_insert_command_table_name(char *query);
static char *skip_comment(char *query);

static POOL_CONNECTION_POOL_SLOT *slots[MAX_CONNECTION_SLOTS];

static int in_load_balance;		/* non 0 if in load balance mode */
static int master_slave_dml;	/* non 0 if master/slave mode is specified in config file */
static int replication_was_enabled;		/* replication mode was enabled */
static int master_slave_was_enabled;	/* master/slave mode was enabled */
static int internal_transaction_started;		/* to issue table lock command a transaction
												   has been started internally */
static int is_drop_database(char *query);		/* returns non 0 if this is a DROP DATABASE command */

POOL_STATUS pool_process_query(POOL_CONNECTION *frontend, 
							   POOL_CONNECTION_POOL *backend,
							   int connection_reuse,
							   int first_ready_for_query_received)
{
	char kind, kind1;	/* packet kind (backend) */
	char fkind;	/* packet kind (frontend) */
	short num_fields = 0;
	fd_set	readmask;
	fd_set	writemask;
	fd_set	exceptmask;
	int fds;
	POOL_STATUS status;
	int state;	/* 0: ok to issue commands 1: waiting for "ready for query" response */
	int qcnt;

	frontend->no_forward = connection_reuse;
	qcnt = 0;
	state = 0;

	for (;;)
	{
		kind = kind1 = 0;
		fkind = 0;

		if (state == 0 && connection_reuse)
		{
			int st;

			/* send query for resetting connection such as "ROLLBACK" "RESET ALL"... */
			st = reset_backend(backend, qcnt);

			if (st < 0)		/* error? */
			{
				/* probably we don't need this, since caller will
				 * close the connection to frontend after returning with POOL_END. But I
				 * guess I would like to be a paranoid...
				 */
				frontend->no_forward = 0;
				return POOL_END;
			}

			else if (st == 0)	/* no query issued? */
			{
				qcnt++;
				continue;
			}

			else if (st == 1)	/* more query remains */
			{
				state = 1;
				qcnt++;
				continue;
			}

			else	/* no more query(st == 2) */
			{
				frontend->no_forward = 0;
				return POOL_CONTINUE;
			}

		}

		if ((!DUAL_MODE && MASTER(backend)->len == 0 && frontend->len == 0) ||
			(DUAL_MODE && MASTER(backend)->len == 0 &&
			SECONDARY(backend)->len == 0
			 && frontend->len == 0))
		{

			struct timeval timeout;

			timeout.tv_sec = 1;
			timeout.tv_usec = 0;

			FD_ZERO(&readmask);
			FD_ZERO(&writemask);
			FD_ZERO(&exceptmask);
			if (!connection_reuse)
				FD_SET(frontend->fd, &readmask);
			FD_SET(MASTER(backend)->fd, &readmask);
			if (DUAL_MODE)
				FD_SET(SECONDARY(backend)->fd, &readmask);
			if (!connection_reuse)
				FD_SET(frontend->fd, &exceptmask);
			FD_SET(MASTER(backend)->fd, &exceptmask);

			if (connection_reuse)
			{
				if (DUAL_MODE)
					fds = select(Max(SECONDARY(backend)->fd, MASTER(backend)->fd) + 1,
								 &readmask, &writemask, &exceptmask, NULL);
				else
					fds = select(MASTER(backend)->fd+1, &readmask, &writemask, &exceptmask, NULL);
			}
			else
			{
				if (DUAL_MODE)
					fds = select(Max(SECONDARY(backend)->fd,
									 Max(frontend->fd, MASTER(backend)->fd)+1),
								 &readmask, &writemask, &exceptmask, NULL);
				else
					fds = select(Max(frontend->fd, MASTER(backend)->fd)+1,
								 &readmask, &writemask, &exceptmask, NULL);
			}

			if (fds == -1)
			{
				if (errno == EINTR)
					continue;

				pool_error("select() failed. reason: %s", strerror(errno));
				return POOL_ERROR;
			}

			if (fds == 0)
			{
				return POOL_CONTINUE;
			}

			if (FD_ISSET(MASTER(backend)->fd, &readmask))
			{
				pool_read(MASTER(backend), &kind, 1);
				pool_debug("read kind from backend %c", kind);
			}

			if (DUAL_MODE && FD_ISSET(SECONDARY(backend)->fd, &readmask))
			{
				pool_read(SECONDARY(backend), &kind1, 1);
				pool_debug("read kind from secondary backend %c", kind1);
			}

			if (!connection_reuse && FD_ISSET(frontend->fd, &exceptmask))
			{
				return POOL_END;
			}
			if (FD_ISSET(MASTER(backend)->fd, &exceptmask))
			{
				return POOL_ERROR;
			}

			if (!connection_reuse && FD_ISSET(frontend->fd, &readmask))
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}
		else
		{
			if (MASTER(backend)->len > 0)
			{
				pool_read(MASTER(backend), &kind, 1);
				if (REPLICATION)
				{
					pool_read(SECONDARY(backend), &kind1, 1);
					if (kind == '\0' || kind != kind1)
					{
						int sts;

						pool_error("pool_process_query: kind does not match between backends master(%c) secondary(%c)",
								   kind, kind1);
						pool_send_error_message(frontend, MAJOR(backend), "XX000", 
												"kind mismatch between backends", "",
												"check data consistency between master and secondary", __FILE__, __LINE__);

						/* health check */
						sts = health_check();
						if (sts == -1)
						{
							notice_backend_error(1);
							exit(1);
						}
						else if (sts == -2)
						{
							notice_backend_error(0);
							exit(1);
						}

						if (pool_config.replication_stop_on_mismatch)
							return POOL_FATAL;
						else
							return POOL_ERROR;
					}
				}
				pool_debug("read kind from backend pending data %c len: %d po: %d", kind, MASTER(backend)->len, MASTER(backend)->po);
			}
			if (frontend->len > 0)
			{
				status = ProcessFrontendResponse(frontend, backend);
				if (status != POOL_CONTINUE)
					return status;

				continue;
			}
		}

		/* this is the synchronous point */
		if (REPLICATION || first_ready_for_query_received)
		{
			if (kind == 0)
			{
				pool_read(MASTER(backend), &kind, 1);
			}
			if (kind1 == 0)
			{
				if (SECONDARY(backend)->len <= 0)
				{
					/* at this point the query should have completed and it's safe to set timeout here */
					pool_debug("pool_process_query: waiting for secondary for data ready");

					/* temporary enable timeout */
					pool_enable_timeout();

					if (pool_check_fd(SECONDARY(backend), 0))
					{
						pool_error("pool_process_query: secondary data is not ready at synchronous point. abort this session");

					}
					else
					{
						pool_read(SECONDARY(backend), &kind1, 1);
					}

					pool_disable_timeout();
				}
				else
				{
						pool_read(SECONDARY(backend), &kind1, 1);
				}
			}

			first_ready_for_query_received = 0;

			if (kind == '\0' || kind != kind1)
			{
				int sts;

				pool_error("pool_process_query: kind does not match between backends master(%c) secondary(%c)",
						   kind, kind1);
				pool_send_error_message(frontend, MAJOR(backend), "XX000", 
										"kind mismatch between backends", "",
										"check data consistency between master and secondary", __FILE__, __LINE__);

				/* health check */
				sts = health_check();
				if (sts == -1)
				{
					notice_backend_error(1);
					exit(1);
				}
				else if (sts == -2)
				{
					notice_backend_error(0);
					exit(1);
				}

				if (pool_config.replication_stop_on_mismatch)
					return POOL_FATAL;
				else
					return POOL_ERROR;
			}
		}

		/*
		 * Prrocess backend Response
		 */

		if (kind == 0)
		{
			pool_error("kind is 0!");
			return POOL_ERROR;
		}

		pool_debug("pool_process_query: kind from backend: %c", kind);

		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			switch (kind)
			{
				case 'G':
					/* CopyIn response */
					status = CopyInResponse(frontend, backend);
					break;
				case 'S':
					/* Paramter Status */
					status = ParameterStatus(frontend, backend);
					break;
				case 'Z':
					/* Ready for query */
					status = ReadyForQuery(frontend, backend, 1);
					break;
				default:
					status = SimpleForwardToFrontend(kind, frontend, backend);
					break;
			}
		}
		else
		{
			switch (kind)
			{
				case 'A':
					/* Notification  response */
					status = NotificationResponse(frontend, backend);
					break;

				case 'B':
					/* BinaryRow */
					status = BinaryRow(frontend, backend, num_fields);
					break;

				case 'C':
					/* Complete command response */
					status = CompleteCommandResponse(frontend, backend);
					break;

				case 'D':
					/* AsciiRow */
					status = AsciiRow(frontend, backend, num_fields);
					break;

				case 'E':
					/* Error Response */
					status = ErrorResponse(frontend, backend);
					break;

				case 'G':
					/* CopyIn Response */
					status = CopyInResponse(frontend, backend);
					break;

				case 'H':
					/* CopyOut Response */
					status = CopyOutResponse(frontend, backend);
					break;

				case 'I':
					/* Empty Query Response */
					status = EmptyQueryResponse(frontend, backend);
					break;

				case 'N':
					/* Notice Response */
					status = NoticeResponse(frontend, backend);
					break;

				case 'P':
					/* CursorResponse */
					status = CursorResponse(frontend, backend);
					break;

				case 'T':
					/* RowDescription */
					status = RowDescription(frontend, backend);
					if (status < 0)
						return POOL_ERROR;

					num_fields = status;
					status = POOL_CONTINUE;
					break;

				case 'V':
					/* FunctionResultResponse and FunctionVoidResponse */
					status = FunctionResultResponse(frontend, backend);
					break;
				
				case 'Z':
					/* Ready for query */
					status = ReadyForQuery(frontend, backend, 1);
					break;
				
				default:
					pool_error("Unknown message type %c(%02x)", kind, kind);
					exit(1);
			}
		}

		if (status != POOL_CONTINUE)
			return status;

		if (kind == 'Z' && frontend->no_forward && state == 1)
		{
			state = 0;
		}

	}
	return POOL_CONTINUE;
}

static POOL_STATUS Query(POOL_CONNECTION *frontend, 
						 POOL_CONNECTION_POOL *backend, char *query)
{
	char *string;
	int len;
	static char *sq = "show pool_status";

	if (query == NULL)	/* need to read query from frontend? */
	{
		/* read actual query */
		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			if (pool_read(frontend, &len, sizeof(len)) < 0)
				return POOL_END;
			len = ntohl(len) - 4;
			string = pool_read2(frontend, len);
		}
		else
			string = pool_read_string(frontend, &len, 0);

		if (string == NULL)
			return POOL_END;
	}
	else
	{
		len = strlen(query)+1;
		string = query;
	}

	pool_debug("Query: %s", string);

	/*
	 * if this is DROP DATABASE command, send HUP signal to parent and
	 * ask it to close all idle connections.
	 * XXX This is overkill. It would be better to close the idle
	 * connection for the database which DROP DATABASE command tries
	 * to drop. This is impossible at this point, since we have no way
	 * to pass such info to other processes.
	 */
	if (is_drop_database(string))
	{
		int stime = 5;	/* XXX give arbitary time to allow closing idle connections */

		pool_debug("Query: sending HUP signal to parent");

		kill(getppid(), SIGHUP);		/* send HUP signal to parent */

		/* we need to loop over here since we will get HUP signal while sleeping */
		while (stime > 0)
		{
			stime = sleep(stime);
		}
	}

	/* process status reporting? */
	if (strncasecmp(sq, string, strlen(sq)) == 0)
	{
		pool_debug("process reporting");
		process_reporting(frontend, backend);
		return POOL_CONTINUE;
	}

	/* load balance trick */
	if (load_balance_enabled(backend, string))
		start_load_balance(backend);
	else if (MASTER_SLAVE)
	{
		master_slave_was_enabled = 1;
		MASTER_SLAVE = 0;
		master_slave_dml = 1;
	}

	/*
	 * judge if we need to lock the table
	 * to keep SERIAL data consistency among servers
	 * conditions:
	 * - protocol is V3
	 * - statement is INSERT
	 * - either "INSERT LOCK" comment exists or insert_lock directive specified
	 */
	if (REPLICATION && need_insert_lock(backend, string))
	{
		/* start a transaction if needed and lock the table */
		if (insert_lock(backend, string) != POOL_CONTINUE)
			return POOL_END;
	}

	/* forward the query to the backend */
	pool_write(MASTER(backend), "Q", 1);

	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		int sendlen = htonl(len + 4);
		pool_write(MASTER(backend), &sendlen, sizeof(sendlen));
	}

	if (pool_write_and_flush(MASTER(backend), string, len) < 0)
	{
		return POOL_END;
	}

	if (REPLICATION)
#ifdef NOT_USED
		(frontend == NULL &&  )		/* we assume that frontend == NULL
								 * means that Query() is called from
								 * reset_backend(). In master/slave
								 * mode AND load balance is enabled,
								 * we need to send reset queries to
								 * secondary as well.
								 */
#endif
	{
		/*
		 * in "strict mode" we need to wait for master completing the query.
		 * note that this is not applied if "NO STRICT" is specified as a comment.
		 */
		if ((pool_config.replication_strict && !NO_STRICT_MODE(string)) ||
			STRICT_MODE(string))
		{
			pool_debug("waiting for master completing the query");
			if (synchronize(MASTER(backend)))
				return POOL_END;
		}

#define SEQUENCE_DEBUG
#ifdef SEQUENCE_DEBUG
		if (!strncmp(string, "/*SLEEP*/", 9))
		{
			pool_debug("start sleeping");
			sleep(20);
			pool_debug("end sleeping");
		}
#endif

		pool_write(SECONDARY(backend), "Q", 1);
		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			int sendlen = htonl(len + 4);
			pool_write(SECONDARY(backend), &sendlen, sizeof(sendlen));
		}

		if (pool_write_and_flush(SECONDARY(backend), string, len) < 0)
		{
			return POOL_END;
		}

		/* in "strict mode" we need to wait for secondary completing the query */
		if (pool_config.replication_strict || STRICT_MODE(string))
			if (synchronize(SECONDARY(backend)))
				return POOL_END;
	}
	return POOL_CONTINUE;
}

/*
 * process EXECUTE (V3 only)
 */
static POOL_STATUS Execute(POOL_CONNECTION *frontend, 
						   POOL_CONNECTION_POOL *backend)
{
	char *string;		/* portal name + null terminate + max_tobe_returned_rows */
	int len;
	int sendlen;
	int i;

	/* read Execute packet */
	if (pool_read(frontend, &len, sizeof(len)) < 0)
		return POOL_END;

	len = ntohl(len) - 4;
	string = pool_read2(frontend, len);

	pool_debug("Execute: portal name <%s>", string);

	for (i = 0;i < backend->num;i++)
	{
		POOL_CONNECTION *cp = backend->slots[i]->con;

		/* forward the query to the backend */
		pool_write(cp, "E", 1);
		sendlen = htonl(len + 4);
		pool_write(cp, &sendlen, sizeof(sendlen));
		pool_write(cp, string, len);

		/*
		 * send "Flush" message so that backend notices us
		 * the completion of the command
		 */
		pool_write(cp, "H", 1);
		sendlen = htonl(4);
		if (pool_write_and_flush(cp, &sendlen, sizeof(sendlen)) < 0)
		{
			return POOL_END;
		}

		if (!REPLICATION)
			break;

		/*
		 * in "strict mode" we need to wait for backend completing the query.
		 */
		if (pool_config.replication_strict)
		{
			pool_debug("waiting for backend[%d] completing the query", i);
			if (synchronize(cp))
				return POOL_END;
		}

	}

	return POOL_CONTINUE;
}

#ifdef NOT_USED
/*
 * process Sync (V3 only)
 */
static POOL_STATUS Sync(POOL_CONNECTION *frontend, 
						   POOL_CONNECTION_POOL *backend)
{
	char *string;		/* portal name + null terminate + max_tobe_returned_rows */
	int len;
	int sendlen;

	/* read Sync packet */
	if (pool_read(frontend, &len, sizeof(len)) < 0)
		return POOL_END;

	len = ntohl(len) - 4;
	string = pool_read2(frontend, len);

	/* forward the query to the backend */
	pool_write(MASTER(backend), "S", 1);

	sendlen = htonl(len + 4);
	pool_write(MASTER(backend), &sendlen, sizeof(sendlen));
	if (pool_write_and_flush(MASTER(backend), string, len) < 0)
	{
		return POOL_END;
	}

	if (REPLICATION)
	{
		/*
		 * in "strict mode" we need to wait for master completing the query.
		 * note that this is not applied if "NO STRICT" is specified as a comment.
		 */
		if (pool_config.replication_strict)
		{
			pool_debug("waiting for master completing the query");
			if (synchronize(MASTER(backend)))
				return POOL_END;
		}

		pool_write(SECONDARY(backend), "S", 1);
		sendlen = htonl(len + 4);
		pool_write(SECONDARY(backend), &sendlen, sizeof(sendlen));
		if (pool_write_and_flush(SECONDARY(backend), string, len) < 0)
		{
			return POOL_END;
		}

		/* in "strict mode" we need to wait for secondary completing the query */
		if (pool_config.replication_strict)
			if (synchronize(SECONDARY(backend)))
				return POOL_END;
	}
	return POOL_CONTINUE;
}
#endif

static POOL_STATUS ReadyForQuery(POOL_CONNECTION *frontend, 
								 POOL_CONNECTION_POOL *backend, int send_ready)
{
	/* if a transaction is started for insert lock, we need to close it. */
	if (internal_transaction_started)
	{
		int i;
		int len;
		signed char state;

		if ((len = pool_read_message_length(backend)) < 0)
			return POOL_END;

		pool_debug("ReadyForQuery: message length: %d", len);

		len = htonl(len);

		state = pool_read_kind(backend);
		if (state < 0)
			return POOL_END;

		/* set transaction state */
		pool_debug("ReadyForQuery: transaction state: %c", state);
		MASTER(backend)->tstate = state;
		if (REPLICATION)
			SECONDARY(backend)->tstate = state;

		for (i = 0;i < backend->num;i++)
		{
			if (do_command(backend->slots[i]->con, "COMMIT", PROTO_MAJOR_V3, 1) != POOL_CONTINUE)
				return POOL_ERROR;
		}
		internal_transaction_started = 0;
	}

	pool_flush(frontend);

	if (send_ready)
	{
		pool_write(frontend, "Z", 1);

		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			int len;
			signed char state;

			if ((len = pool_read_message_length(backend)) < 0)
				return POOL_END;

			pool_debug("ReadyForQuery: message length: %d", len);

			len = htonl(len);
			pool_write(frontend, &len, sizeof(len));

			state = pool_read_kind(backend);
			if (state < 0)
				return POOL_END;

			/* set transaction state */
			pool_debug("ReadyForQuery: transaction state: %c", state);
			MASTER(backend)->tstate = state;
			if (REPLICATION)
				SECONDARY(backend)->tstate = state;

			pool_write(frontend, &state, 1);
		}

		if (pool_flush(frontend))
			return POOL_END;
	}

	/* end load balance mode */
	if (in_load_balance)
		end_load_balance(backend);

	if (master_slave_dml)
	{
		MASTER_SLAVE = 1;
		master_slave_was_enabled = 0;
		master_slave_dml = 0;
	}

#ifdef NOT_USED
	return ProcessFrontendResponse(frontend, backend);
#endif
	return POOL_CONTINUE;
}

static POOL_STATUS CompleteCommandResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read command tag */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;

	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;

		if (len != len1)
		{
			pool_debug("Complete Command Response: message length does not match between master(%d \"%s\",) and secondary(%d \"%s\",)",
					 len, string, len1, string1);
		}
	}

	/* forward to the frontend */
	pool_write(frontend, "C", 1);
	pool_debug("Complete Command Response: string: \"%s\"", string);
	if (pool_write(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static int RowDescription(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend)
{
	short num_fields, num_fields1;
	int oid, mod;
	int oid1, mod1;
	short size, size1;
	char *string, *string1;
	int len, len1;
	int i;

	/* # of fields (could be 0) */
	pool_read(MASTER(backend), &num_fields, sizeof(short));
	if (REPLICATION)
	{
		pool_read(SECONDARY(backend), &num_fields1, sizeof(short));
		if (num_fields != num_fields1)
		{
			pool_error("RowDescription: num_fields deos not match between backends master(%d) and secondary(%d)",
					   num_fields, num_fields1);
			return POOL_FATAL;
		}
	}

	/* forward it to the frontend */
	pool_write(frontend, "T", 1);
	pool_write(frontend, &num_fields, sizeof(short));

	num_fields = ntohs(num_fields);
	for (i = 0;i<num_fields;i++)
	{
		/* field name */
		string = pool_read_string(MASTER(backend), &len, 0);
		if (string == NULL)
			return POOL_END;

		if (REPLICATION)
		{
			string1 = pool_read_string(SECONDARY(backend), &len1, 0);
			if (string == NULL)
				return POOL_END;
			if (len != len1)
			{
				pool_error("RowDescription: field length deos not match between backends master(%d) and secondary(%d)",
						   ntohl(len), ntohl(len1));
				return POOL_FATAL;
			}
		}

		pool_write(frontend, string, len);

		/* type oid */
		pool_read(MASTER(backend), &oid, sizeof(int));

		pool_debug("RowDescription: type oid: %d", ntohl(oid));

		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &oid1, sizeof(int));

			/* we do not regard oid mismatch as fatal */
			if (oid != oid1)
			{
				pool_debug("RowDescription: field oid deos not match between backends master(%d) and secondary(%d)",
						 ntohl(oid), ntohl(oid1));
			}
		}
		pool_write(frontend, &oid, sizeof(int));

		/* size */
		pool_read(MASTER(backend), &size, sizeof(short));
		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &size1, sizeof(short));
			if (size1 != size1)
			{
				pool_error("RowDescription: field size deos not match between backends master(%d) and secondary(%d)",
						 ntohs(size), ntohs(size1));
				return POOL_FATAL;
			}
		}
		pool_debug("RowDescription: field size: %d", ntohs(size));
		pool_write(frontend, &size, sizeof(short));

		/* modifier */
		pool_read(MASTER(backend), &mod, sizeof(int));

		pool_debug("RowDescription: modifier: %d", ntohs(mod));

		if (REPLICATION)
		{
			pool_read(SECONDARY(backend), &mod1, sizeof(int));
			if (mod != mod1)
			{
				pool_debug("RowDescription: modifier deos not match between backends master(%d) and secondary(%d)",
						 ntohl(mod), ntohl(mod1));
			}
		}
		pool_write(frontend, &mod, sizeof(int));
	}

	return num_fields;
}

static POOL_STATUS AsciiRow(POOL_CONNECTION *frontend, 
							POOL_CONNECTION_POOL *backend,
							short num_fields)
{
	static char nullmap[8192], nullmap1[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size, size1;
	char *buf;
	char msgbuf[1024];

	pool_write(frontend, "D", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(MASTER(backend), nullmap, nbytes);
	if (pool_write(frontend, nullmap, nbytes) < 0)
		return POOL_END;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), nullmap1, nbytes) < 0)
			return POOL_END;

		if (memcmp(nullmap, nullmap1, nbytes))
		{
			/* XXX: NULLMAP maybe different among
			   backends. If we were a paranoid, we have to treat
			   this as a fatal error. However in the real world
			   we'd better to adapt this situation. Just throw a
			   log... */
			pool_debug("AsciiRow: NULLMAP differ between master and secondary");
		}
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			if (pool_read(MASTER(backend), &size, sizeof(int)) < 0)
				return POOL_END;
		}

		if (REPLICATION && (mask & nullmap1[i/8]))
		{
			/* XXX: field size maybe different among
			   backends. If we were a paranoid, we have to treat
			   this as a fatal error. However in the real world
			   we'd better to adapt this situation. Just throw a
			   log... */

			if (pool_read(SECONDARY(backend), &size1, sizeof(int)) < 0)
				return POOL_END;

			if (size != size1)
				pool_debug("AsciiRow: %d th field size does not match between master(%d) and secondary(%d)",
						 i, ntohl(size), ntohl(size1));
			size1 = ntohl(size1) - 4;
		}

		buf = NULL;

		if (mask & nullmap[i/8])
		{
			/* forward to frontend */
			pool_write(frontend, &size, sizeof(int));
			size = ntohl(size) - 4;

			/* read and send actual data only when size > 0 */
			if (size > 0)
			{
				buf = pool_read2(MASTER(backend), size);
				if (buf == NULL)
					return POOL_END;
			}
		}

		if (REPLICATION && size1 > 0 && (mask & nullmap1[i/8]))
		{
			/* read and discard secondary data */
			if (pool_read2(SECONDARY(backend), size1) == NULL)
				return POOL_END;
		}

		if (buf)
		{
			pool_write(frontend, buf, size);
			snprintf(msgbuf, Min(sizeof(msgbuf), size+1), "%s", buf);
			pool_debug("AsciiRow: len: %d data: %s", size, msgbuf);
		}

		mask >>= 1;
	}

	return POOL_CONTINUE;
}

static POOL_STATUS BinaryRow(POOL_CONNECTION *frontend, 
							 POOL_CONNECTION_POOL *backend,
							 short num_fields)
{
	static char nullmap[8192], nullmap1[8192];
	int nbytes;
	int i;
	unsigned char mask;
	int size, size1;
	char *buf;

	pool_write(frontend, "B", 1);

	nbytes = (num_fields + 7)/8;

	if (nbytes <= 0)
		return POOL_CONTINUE;

	/* NULL map */
	pool_read(MASTER(backend), nullmap, nbytes);
	if (pool_write(frontend, nullmap, nbytes) < 0)
		return POOL_END;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), nullmap1, nbytes) < 0)
			return POOL_END;

		if (memcmp(nullmap, nullmap1, nbytes))
		{
			/* XXX: NULLMAP maybe different among
			   backends. If we were a paranoid, we have to treat
			   this as a fatal error. However in the real world
			   we'd better to adapt this situation. Just throw a
			   log... */
			pool_debug("BinaryRow: NULLMAP differ between master and secondary");
		}
	}

	mask = 0;

	for (i = 0;i<num_fields;i++)
	{
		if (mask == 0)
			mask = 0x80;

		/* NOT NULL? */
		if (mask & nullmap[i/8])
		{
			/* field size */
			if (pool_read(MASTER(backend), &size, sizeof(int)) < 0)
				return POOL_END;
		}

		if (REPLICATION && (mask & nullmap1[i/8]))
		{
			/* XXX: field size maybe different among
			   backends. If we were a paranoid, we have to treat
			   this as a fatal error. However in the real world
			   we'd better to adapt this situation. Just throw a
			   log... */

			if (pool_read(SECONDARY(backend), &size1, sizeof(int)) < 0)
				return POOL_END;

			if (size != size1)
				pool_debug("BinaryRow: %d th field size does not match between master(%d) and secondary(%d)",
						 i, ntohl(size), ntohl(size1));
			size1 = ntohl(size1) - 4;
		}

		buf = NULL;

		if (mask & nullmap[i/8])
		{
			/* forward to frontend */
			pool_write(frontend, &size, sizeof(int));
			size = ntohl(size) - 4;

			/* read and send actual data only when size > 0 */
			if (size > 0)
			{
				buf = pool_read2(MASTER(backend), size);
				if (buf == NULL)
					return POOL_END;
			}
		}

		if (REPLICATION && size1 > 0 && (mask & nullmap1[i/8]))
		{
			/* read and discard secondary data */
			if (pool_read2(SECONDARY(backend), size1) == NULL)
				return POOL_END;
		}

		if (buf)
			pool_write(frontend, buf, size);

		mask >>= 1;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS CursorResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read cursor name */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;
		if (len != len1)
		{
			pool_error("CursorResponse: length does not match between master(%d) and secondary(%d)",
					   len, len1);
			pool_error("CursorResponse: master(%s) secondary(%s)", string, string1);
			return POOL_END;
		}
	}

	/* forward to the frontend */
	pool_write(frontend, "P", 1);
	if (pool_write(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

POOL_STATUS ErrorResponse(POOL_CONNECTION *frontend, 
						  POOL_CONNECTION_POOL *backend)
{
	char *string;
	int len;

	/* read error message */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string = pool_read_string(SECONDARY(backend), &len, 0);
		if (string == NULL)
			return POOL_END;
	}

	/* forward to the frontend */
	pool_write(frontend, "E", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
		return POOL_END;
			
	return POOL_CONTINUE;
}

POOL_STATUS NoticeResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	char *string, *string1;
	int len, len1;

	/* read notice message */
	string = pool_read_string(MASTER(backend), &len, 0);
	if (string == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		string1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (string1 == NULL)
			return POOL_END;
	}

	/* forward to the frontend */
	pool_write(frontend, "N", 1);
	if (pool_write_and_flush(frontend, string, len) < 0)
	{
		return POOL_END;
	}
	return POOL_CONTINUE;
}

static POOL_STATUS CopyInResponse(POOL_CONNECTION *frontend, 
								  POOL_CONNECTION_POOL *backend)
{
	POOL_STATUS status;

	/* forward to the frontend */
	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		if (SimpleForwardToFrontend('G', frontend, backend) != POOL_CONTINUE)
			return POOL_END;
		if (pool_flush(frontend) != POOL_CONTINUE)
			return POOL_END;
	}
	else
		if (pool_write_and_flush(frontend, "G", 1) < 0)
			return POOL_END;

	status = CopyDataRows(frontend, backend, 1);
	return status;
}

static POOL_STATUS CopyOutResponse(POOL_CONNECTION *frontend, 
								   POOL_CONNECTION_POOL *backend)
{
	POOL_STATUS status;

	/* forward to the frontend */
	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		if (SimpleForwardToFrontend('H', frontend, backend) != POOL_CONTINUE)
			return POOL_END;
		if (pool_flush(frontend) != POOL_CONTINUE)
			return POOL_END;
	}
	else
		if (pool_write_and_flush(frontend, "H", 1) < 0)
			return POOL_END;

	status = CopyDataRows(frontend, backend, 0);
	return status;
}

static POOL_STATUS CopyDataRows(POOL_CONNECTION *frontend,
								POOL_CONNECTION_POOL *backend, int copyin)
{
	char *string;
	int len;

#ifdef DEBUG
	int i = 0;
	char buf[1024];
#endif

	for (;;)
	{
		if (copyin)
		{
			if (MAJOR(backend) == PROTO_MAJOR_V3)
			{
				char kind;

				if (pool_read(frontend, &kind, 1) < 0)
					return POOL_END;
				
				SimpleForwardToBackend(kind, frontend, backend);

				/* CopyData? */
				if (kind == 'd')
					continue;
				else
					break;
			}
			else
				string = pool_read_string(frontend, &len, 1);
		}
		else
		{
			/* CopyOut */
			if (MAJOR(backend) == PROTO_MAJOR_V3)
			{
				signed char kind;

				if ((kind = pool_read_kind(backend)) < 0)
					return POOL_END;
				
				SimpleForwardToFrontend(kind, frontend, backend);

				/* CopyData? */
				if (kind == 'd')
					continue;
				else
					break;
			}
			else
			{
				string = pool_read_string(MASTER(backend), &len, 1);
				if (REPLICATION)
					string = pool_read_string(SECONDARY(backend), &len, 1);
			}
		}

		if (string == NULL)
			return POOL_END;

#ifdef DEBUG
		strncpy(buf, string, len);
		pool_debug("copy line %d %d bytes :%s:", i++, len, buf);
#endif

		if (copyin)
		{
			pool_write(MASTER(backend), string, len);
			if (REPLICATION)
				pool_write(SECONDARY(backend), string, len);
		}
		else
			pool_write(frontend, string, len);			

		if (len == PROTO_MAJOR_V3)
		{
			/* end of copy? */
			if (string[0] == '\\' &&
				string[1] == '.' &&
				string[2] == '\n')
			{
				break;
			}
		}
	}

	if (copyin)
	{
		if (pool_flush(MASTER(backend)) <0)
			return POOL_END;
		if (REPLICATION)
		{
			if (pool_flush(SECONDARY(backend)) <0)
				return POOL_END;
		}
	}
	else
		if (pool_flush(frontend) <0)
			return POOL_END;

	return POOL_CONTINUE;
}

static POOL_STATUS EmptyQueryResponse(POOL_CONNECTION *frontend,
									  POOL_CONNECTION_POOL *backend)
{
	char c;

	if (pool_read(MASTER(backend), &c, sizeof(c)) < 0)
		return POOL_END;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), &c, sizeof(c)) < 0)
			return POOL_END;
	}

	pool_write(frontend, "I", 1);
	return pool_write_and_flush(frontend, "", 1);
}

static POOL_STATUS NotificationResponse(POOL_CONNECTION *frontend, 
										POOL_CONNECTION_POOL *backend)
{
	int pid, pid1;
	char *condition, *condition1;
	int len, len1;

	pool_write(frontend, "A", 1);

	if (pool_read(MASTER(backend), &pid, sizeof(pid)) < 0)
		return POOL_ERROR;

	if (REPLICATION)
	{
		if (pool_read(SECONDARY(backend), &pid1, sizeof(pid1)) < 0)
			return POOL_ERROR;
	}

	condition = pool_read_string(MASTER(backend), &len, 0);
	if (condition == NULL)
		return POOL_END;
	if (REPLICATION)
	{
		condition1 = pool_read_string(SECONDARY(backend), &len1, 0);
		if (condition1 == NULL)
			return POOL_END;
	}

	pool_write(frontend, &pid, sizeof(pid));

	return pool_write_and_flush(frontend, condition, len);
}

static POOL_STATUS FunctionCall(POOL_CONNECTION *frontend, 
								POOL_CONNECTION_POOL *backend)
{
	char dummy[2];
	int oid;
	int argn;
	int i;

	pool_write(MASTER(backend), "F", 1);
	if (REPLICATION)
		pool_write(SECONDARY(backend), "F", 1);

	/* dummy */
	if (pool_read(frontend, dummy, sizeof(dummy)) < 0)
		return POOL_ERROR;
	pool_write(MASTER(backend), dummy, sizeof(dummy));
	if (REPLICATION)
		pool_write(SECONDARY(backend), dummy, sizeof(dummy));

	/* function object id */
	if (pool_read(frontend, &oid, sizeof(oid)) < 0)
		return POOL_ERROR;

	pool_write(MASTER(backend), &oid, sizeof(oid));
	if (REPLICATION)
		pool_write(SECONDARY(backend), &oid, sizeof(oid));

	/* number of arguments */
	if (pool_read(frontend, &argn, sizeof(argn)) < 0)
		return POOL_ERROR;
	pool_write(MASTER(backend), &argn, sizeof(argn));
	if (REPLICATION)
		pool_write(SECONDARY(backend), &argn, sizeof(argn));

	argn = ntohl(argn);

	for (i=0;i<argn;i++)
	{
		int len;
		char *arg;

		/* length of each argument in bytes */
		if (pool_read(frontend, &len, sizeof(len)) < 0)
			return POOL_ERROR;

		pool_write(MASTER(backend), &len, sizeof(len));
		if (REPLICATION)
			pool_write(SECONDARY(backend), &len, sizeof(len));

		len = ntohl(len);

		/* argument value itself */
		if ((arg = pool_read2(frontend, len)) == NULL)
			return POOL_ERROR;
		pool_write(MASTER(backend), arg, len);
		if (REPLICATION)
			pool_write(SECONDARY(backend), arg, len);
	}

	if (pool_flush(MASTER(backend)))
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_flush(SECONDARY(backend)))
			return POOL_ERROR;
	return POOL_CONTINUE;
}

static POOL_STATUS FunctionResultResponse(POOL_CONNECTION *frontend, 
										  POOL_CONNECTION_POOL *backend)
{
	char dummy;
	int len;
	char *result;

	pool_write(frontend, "V", 1);

	if (pool_read(MASTER(backend), &dummy, 1) < 0)
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_read(SECONDARY(backend), &dummy, 1) < 0)
			return POOL_ERROR;

	pool_write(frontend, &dummy, 1);

	/* non empty result? */
	if (dummy == 'G')
	{
		/* length of result in bytes */
		if (pool_read(MASTER(backend), &len, sizeof(len)) < 0)
			return POOL_ERROR;
		if (REPLICATION)
			if (pool_read(SECONDARY(backend), &len, sizeof(len)) < 0)
				return POOL_ERROR;

		pool_write(frontend, &len, sizeof(len));

		len = ntohl(len);

		/* result value itself */
		if ((result = pool_read2(MASTER(backend), len)) == NULL)
			return POOL_ERROR;
		if (REPLICATION)
			if (pool_read(SECONDARY(backend), result, len) < 0)
				return POOL_ERROR;

		pool_write(frontend, result, len);
	}

	/* unused ('0') */
	if (pool_read(MASTER(backend), &dummy, 1) < 0)
		return POOL_ERROR;
	if (REPLICATION)
		if (pool_read(SECONDARY(backend), &dummy, 1) < 0)
			return POOL_ERROR;

	pool_write(frontend, "0", 1);

	return pool_flush(frontend);
}

static POOL_STATUS ProcessFrontendResponse(POOL_CONNECTION *frontend, 
										   POOL_CONNECTION_POOL *backend)
{
	char fkind;
	POOL_STATUS status;

	if (frontend->len <= 0 && frontend->no_forward != 0)
		return POOL_CONTINUE;

	if (pool_read(frontend, &fkind, 1) < 0)
	{
		pool_error("ProcessFrontendResponse: failed to read kind from frontend. fronend abnormally exited");
		return POOL_END;
	}

	pool_debug("read kind from frontend %c(%02x)", fkind, fkind);

	switch (fkind)
	{
		case 'X':
			if (MAJOR(backend) == PROTO_MAJOR_V3)
			{
				int len;
				pool_read(frontend, &len, sizeof(len));
			}
			status = POOL_END;
			break;

		case 'Q':
			status = Query(frontend, backend, NULL);
			break;

/*
		case 'S':
			status = Sync(frontend, backend);
			break;
*/

		case 'E':
			status = Execute(frontend, backend);
		break;

		default:
			if (MAJOR(backend) == PROTO_MAJOR_V3)
			{
				if (MASTER_SLAVE)
				{
					master_slave_was_enabled = 1;
					MASTER_SLAVE = 0;
					master_slave_dml = 1;
				}

				status = SimpleForwardToBackend(fkind, frontend, backend);
				if (pool_flush(MASTER(backend)))
					status = POOL_ERROR;
				if (REPLICATION)
					if (pool_flush(SECONDARY(backend)))
						status = POOL_ERROR;
			}
			else if (MAJOR(backend) == PROTO_MAJOR_V2 && fkind == 'F')
				status = FunctionCall(frontend, backend);
			else
			{
				pool_error("ProcessFrontendResponse: unknown message type %c(%02x)", fkind, fkind);
				status = POOL_ERROR;
			}
			break;
	}

	return status;
}

static int timeoutmsec;

/*
 * enable read timeout
 */
void pool_enable_timeout()
{
	timeoutmsec = pool_config.replication_timeout;
}

/*
 * disable read timeout
 */
void pool_disable_timeout()
{
	timeoutmsec = 0;
}

/*
 * wait until read data is ready
 */
static int synchronize(POOL_CONNECTION *cp)
{
	return pool_check_fd(cp, 1);
}

/*
 * wait until read data is ready
 * if notimeout is non 0, wait forever.
 */
int pool_check_fd(POOL_CONNECTION *cp, int notimeout)
{
	fd_set readmask;
	fd_set exceptmask;
	int fd;
	int fds;
	struct timeval timeout;
	struct timeval *tp;

	fd = cp->fd;

	for (;;)
	{
		FD_ZERO(&readmask);
		FD_ZERO(&exceptmask);
		FD_SET(fd, &readmask);
		FD_SET(fd, &exceptmask);

		if (notimeout || timeoutmsec == 0)
			tp = NULL;
		else
		{
			timeout.tv_sec = pool_config.replication_timeout / 1000;
			timeout.tv_usec = (pool_config.replication_timeout - (timeout.tv_sec * 1000))*1000;
			tp = &timeout;
		}

		fds = select(fd+1, &readmask, NULL, &exceptmask, tp);

		if (fds == -1)
		{
			if (errno == EAGAIN || errno == EINTR)
				continue;

			pool_error("pool_check_fd: select() failed. reason %s", strerror(errno));
			break;
		}

		if (FD_ISSET(fd, &exceptmask))
		{
			pool_error("pool_check_fd: exception occurred");
			break;
		}

		if (fds == 0)
		{
			pool_error("pool_check_fd: data is not ready tp->tv_sec %d tp->tp_usec %d", 
					   pool_config.replication_timeout / 1000,
					   (pool_config.replication_timeout - (timeout.tv_sec * 1000))*1000);
			break;
		}
		return 0;
	}
	return -1;
}

static void process_reporting(POOL_CONNECTION *frontend, POOL_CONNECTION_POOL *backend)
{
	static char *cursorname = "blank";
	static short num_fields = 3;
	static char *field_names[] = {"item", "value", "description"};
	static int oid = 0;
	static short fsize = -1;
	static int mod = 0;
	short n;
	int i, j;
	short s;
	int len;
	short colnum;

	static char nullmap[2] = {0xff, 0xff};
	int nbytes = (num_fields + 7)/8;

#define MAXVALLEN 512

	typedef struct {
		char *name;
		char value[MAXVALLEN+1];
		char *desc;
	} POOL_REPORT_STATUS;

#define MAXITEMS 128

	POOL_REPORT_STATUS status[MAXITEMS];

	short nrows;
	int size;
	int hsize;
	int slen;

	i = 0;

	status[i].name = "listen_addresses";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.listen_addresses);
	status[i].desc = "host name(s) or IP address(es) to listen to";
	i++;

	status[i].name = "port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.port);
	status[i].desc = "pgpool accepting port number";
	i++;

	status[i].name = "socket_dir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.socket_dir);
	status[i].desc = "pgpool socket directory";
	i++;

	status[i].name = "backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.backend_host_name);
	status[i].desc = "master backend host name";
	i++;

	status[i].name = "backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.backend_port);
	status[i].desc = "master backend port number";
	i++;

	status[i].name = "secondary_backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.secondary_backend_host_name);
	status[i].desc = "secondary backend host name";
	i++;

	status[i].name = "secondary_backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.secondary_backend_port);
	status[i].desc = "secondary backend port number";
	i++;

	status[i].name = "num_init_children";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.num_init_children);
	status[i].desc = "# of children initially pre-forked";
	i++;

	status[i].name = "child_life_time";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.child_life_time);
	status[i].desc = "if idle for this seconds, child exits";
	i++;

	status[i].name = "connection_life_time";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.connection_life_time);
	status[i].desc = "if idle for this seconds, connection closes";
	i++;

	status[i].name = "child_max_connections";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.child_max_connections);
	status[i].desc = "if max_connections received, chile exits";
	i++;

	status[i].name = "max_pool";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.max_pool);
	status[i].desc = "max # of connection pool per child";
	i++;

	status[i].name = "logdir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.logdir);
	status[i].desc = "logging directory";
	i++;

	status[i].name = "backend_socket_dir";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.backend_socket_dir);
	status[i].desc = "Unix domain socket directory for the PostgreSQL server";
	i++;

	status[i].name = "replication_mode";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_mode);
	status[i].desc = "non 0 if operating in replication mode";
	i++;

	status[i].name = "replication_strict";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_strict);
	status[i].desc = "non 0 if operating in strict mode";
	i++;

	status[i].name = "replication_timeout";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_timeout);
	status[i].desc = "if secondary does not respond in this milli seconds, abort the session";
	i++;

	status[i].name = "load_balance_mode";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.load_balance_mode);
	status[i].desc = "non 0 if operating in load balancing mode";
	i++;

	status[i].name = "weight_master";
	snprintf(status[i].value, MAXVALLEN, "%f", pool_config.weight_master);
	status[i].desc = "weight of master";
	i++;

	status[i].name = "weight_secondary";
	snprintf(status[i].value, MAXVALLEN, "%f", pool_config.weight_secondary);
	status[i].desc = "weight of secondary";
	i++;

	status[i].name = "replication_stop_on_mismatch";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_stop_on_mismatch);
	status[i].desc = "stop replication mode on fatal error";
	i++;

	status[i].name = "reset_query_list";
	*(status[i].value) = '\0';
	for (j=0;j<pool_config.num_reset_queries;j++)
	{
		int len;
		len = MAXVALLEN - strlen(status[i].value);
		strncat(status[i].value, pool_config.reset_query_list[j], len);
		len = MAXVALLEN - strlen(status[i].value);
		strncat(status[i].value, ";", len);
	}
	status[i]. desc = "queries issued at the end of session";
	i++;

	status[i].name = "print_timestamp";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.print_timestamp);
	status[i].desc = "if true print time stamp to each log line";
	i++;

	status[i].name = "master_slave_mode";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.master_slave_mode);
	status[i].desc = "if true, operate in master/slave mode";
	i++;
		 
	status[i].name = "connection_cache";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.connection_cache);
	status[i].desc = "if true, cache connection pool";
	i++;

	status[i].name = "health_check_timeout";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.health_check_timeout);
	status[i].desc = "health check timeout";
	i++;

	status[i].name = "health_check_period";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.health_check_period);
	status[i].desc = "health check period";
	i++;

	status[i].name = "health_check_user";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.health_check_user);
	status[i].desc = "health check user";
	i++;

	status[i].name = "insert_lock";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.insert_lock);
	status[i].desc = "insert lock";
	i++;

	status[i].name = "ignore_leading_white_space";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.ignore_leading_white_space);
	status[i].desc = "ignore leading white spaces";
	i++;

	status[i].name = "current_backend_host_name";
	snprintf(status[i].value, MAXVALLEN, "%s", pool_config.current_backend_host_name);
	status[i].desc = "current master host name";
	i++;

	status[i].name = "current_backend_port";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.current_backend_port);
	status[i].desc = "current master port #";
	i++;

	status[i].name = "replication_enabled";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.replication_enabled);
	status[i].desc = "non 0 if actually operating in replication mode";
	i++;

	status[i].name = "master_slave_enabled";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.master_slave_enabled);
	status[i].desc = "non 0 if actually operating in master/slave";
	i++;

	status[i].name = "num_reset_queries";
	snprintf(status[i].value, MAXVALLEN, "%d", pool_config.num_reset_queries);
	status[i].desc = "number of queries in reset_query_list";
	i++;

	status[i].name = "server_status";

	if (pool_config.server_status[0] == 0)
	{
		snprintf(status[i].value, MAXVALLEN, "master(%s on %d) unused ",
		  pool_config.backend_host_name, pool_config.backend_port);
	}
	else if (pool_config.server_status[0] == 1)
	{
		snprintf(status[i].value, MAXVALLEN, "master(%s on %d) up ",
		  pool_config.backend_host_name, pool_config.backend_port);
	}
	else if (pool_config.server_status[0] == 2)
	{
		snprintf(status[i].value, MAXVALLEN, "master(%s on %d) down ",
		  pool_config.backend_host_name, pool_config.backend_port);
	}

	slen = strlen(status[i].value);

	if (pool_config.server_status[1] == 0)
	{
		snprintf(status[i].value+slen, MAXVALLEN-slen, "secondary(%s on %d) unused",
		  pool_config.secondary_backend_host_name, pool_config.secondary_backend_port);
	}
	else if (pool_config.server_status[1] == 1)
	{
		snprintf(status[i].value+slen, MAXVALLEN-slen, "secondary(%s on %d) up",
		  pool_config.secondary_backend_host_name, pool_config.secondary_backend_port);
	}
	else if (pool_config.server_status[1] == 2)
	{
		snprintf(status[i].value+slen, MAXVALLEN-slen, "secondary(%s on %d) down",
		  pool_config.secondary_backend_host_name, pool_config.secondary_backend_port);
	}
	status[i].desc = "server status";
	i++;

	nrows = i;

	if (MAJOR(backend) == PROTO_MAJOR_V2)
	{
		/* cursor response */
		pool_write(frontend, "P", 1);
		pool_write(frontend, cursorname, strlen(cursorname)+1);
	}

	/* row description */
	pool_write(frontend, "T", 1);

	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		len = sizeof(num_fields) + sizeof(len);

		for (i=0;i<num_fields;i++)
		{
			char *f = field_names[i];
			len += strlen(f)+1;
			len += sizeof(oid);
			len += sizeof(colnum);
			len += sizeof(oid);
			len += sizeof(s);
			len += sizeof(mod);
			len += sizeof(s);
		}

		len = htonl(len);
		pool_write(frontend, &len, sizeof(len));
	}

	n = htons(num_fields);
	pool_write(frontend, &n, sizeof(short));

	for (i=0;i<num_fields;i++)
	{
		char *f = field_names[i];

		pool_write(frontend, f, strlen(f)+1);		/* field name */

		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			pool_write(frontend, &oid, sizeof(oid));	/* table oid */
			colnum = htons(i);
			pool_write(frontend, &colnum, sizeof(colnum));	/* column number */
		}

		pool_write(frontend, &oid, sizeof(oid));		/* data type oid */
		s = htons(fsize);
		pool_write(frontend, &s, sizeof(fsize));		/* field size */
		pool_write(frontend, &mod, sizeof(mod));		/* modifier */

		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			s = htons(0);
			pool_write(frontend, &s, sizeof(fsize));	/* field format (text) */
		}
	}
	pool_flush(frontend);

	if (MAJOR(backend) == PROTO_MAJOR_V2)
	{
		/* ascii row */
		for (i=0;i<nrows;i++)
		{
			pool_write(frontend, "D", 1);
			pool_write_and_flush(frontend, nullmap, nbytes);

			size = strlen(status[i].name);
			hsize = htonl(size+4);
			pool_write(frontend, &hsize, sizeof(hsize));
			pool_write(frontend, status[i].name, size);

			size = strlen(status[i].value);
			hsize = htonl(size+4);
			pool_write(frontend, &hsize, sizeof(hsize));
			pool_write(frontend, status[i].value, size);

			size = strlen(status[i].desc);
			hsize = htonl(size+4);
			pool_write(frontend, &hsize, sizeof(hsize));
			pool_write(frontend, status[i].desc, size);
		}
	}
	else
	{
		/* data row */
		for (i=0;i<nrows;i++)
		{
			pool_write(frontend, "D", 1);
			len = sizeof(len) + sizeof(nrows);
			len += sizeof(int) + strlen(status[i].name);
			len += sizeof(int) + strlen(status[i].value);
			len += sizeof(int) + strlen(status[i].desc);
			len = htonl(len);
			pool_write(frontend, &len, sizeof(len));
			s = htons(3);
			pool_write(frontend, &s, sizeof(s));

			len = htonl(strlen(status[i].name));
			pool_write(frontend, &len, sizeof(len));
			pool_write(frontend, status[i].name, strlen(status[i].name));

			len = htonl(strlen(status[i].value));
			pool_write(frontend, &len, sizeof(len));
			pool_write(frontend, status[i].value, strlen(status[i].value));
			
			len = htonl(strlen(status[i].desc));
			pool_write(frontend, &len, sizeof(len));
			pool_write(frontend, status[i].desc, strlen(status[i].desc));
		}
	}

	/* complete command response */
	pool_write(frontend, "C", 1);
	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		len = htonl(sizeof(len) + strlen("SELECT")+1);
		pool_write(frontend, &len, sizeof(len));
	}
	pool_write(frontend, "SELECT", strlen("SELECT")+1);

	/* ready for query */
	pool_write(frontend, "Z", 1);
	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		len = htonl(sizeof(len) + 1);
		pool_write(frontend, &len, sizeof(len));
		pool_write(frontend, "I", 1);
	}

	pool_flush(frontend);
}

void pool_send_frontend_exits(POOL_CONNECTION_POOL *backend)
{
	int len;

	pool_write(MASTER(backend), "X", 1);

	if (MAJOR(backend) == PROTO_MAJOR_V3)
	{
		len = htonl(4);
		pool_write(MASTER(backend), &len, sizeof(len));
	}

	/*
	 * XXX we cannot call pool_flush() here since backend may already
	 * close the socket and pool_flush() automatically invokes fail
	 * over handler. This could happen in copy command (remember the
	 * famouse "lost synchronization with server, resettin g
	 * connection" message)
	 */
	pool_flush_it(MASTER(backend));

	if (DUAL_MODE)
	{
		pool_write(SECONDARY(backend), "X", 1);
		if (MAJOR(backend) == PROTO_MAJOR_V3)
		{
			len = htonl(4);
			pool_write(SECONDARY(backend), &len, sizeof(len));
		}
		pool_flush_it(SECONDARY(backend));
	}
}

/*
 * -------------------------------------------------------
 * V3 functions
 * -------------------------------------------------------
 */
POOL_STATUS SimpleForwardToFrontend(char kind, POOL_CONNECTION *frontend, POOL_CONNECTION_POOL *backend)
{
	int len, len1;
	char *p;
	int status;

	pool_write(frontend, &kind, 1);

	status = pool_read(MASTER(backend), &len, sizeof(len));
	if (status < 0)
	{
		pool_error("SimpleForwardToFrontend: error while reading message length");
		return POOL_END;
	}

	if (REPLICATION)
	{
		status = pool_read(SECONDARY(backend), &len1, sizeof(len1));
		if (status < 0)
		{
			pool_error("SimpleForwardToFrontend: error while reading message length from secondary backend");
			return POOL_END;
		}

		if (len != len1)
		{
			pool_debug("SimpleForwardToFrontend: length does not match between backends master(%d) secondary(%d) kind:(%c)",
					 ntohl(len), ntohl(len1), kind);
		}
	}

	pool_write(frontend, &len, sizeof(len));

	len = ntohl(len);
	len -= 4;

	p = pool_read2(MASTER(backend), len);
	if (p == NULL)
		return POOL_END;

	if (REPLICATION)
	{
		len1 = ntohl(len1);
		len1 -= 4;
		if (pool_read2(SECONDARY(backend), len1) == NULL)
			return POOL_END;
	}

	return pool_write(frontend, p, len);
}

POOL_STATUS SimpleForwardToBackend(char kind, POOL_CONNECTION *frontend, POOL_CONNECTION_POOL *backend)
{
	int len;
	int sendlen;
	char *p;

	if (pool_write(MASTER(backend), &kind, 1))
		return POOL_END;
	if (REPLICATION)
		if (pool_write(SECONDARY(backend), &kind, 1))
			return POOL_END;

	if (pool_read(frontend, &sendlen, sizeof(sendlen)))
	{
		return POOL_END;
	}

	len = ntohl(sendlen) - 4;

	p = pool_read2(frontend, len);
	if (p == NULL)
		return POOL_END;

	if (pool_write(MASTER(backend), &sendlen, sizeof(sendlen)))
		return POOL_END;
	if (pool_write(MASTER(backend), p, len))
		return POOL_END;

	if (REPLICATION)
	{
		if (pool_write(SECONDARY(backend), &sendlen, sizeof(sendlen)))
			return POOL_END;
		if (pool_write(SECONDARY(backend), p, len))
			return POOL_END;
	}

	return POOL_CONTINUE;
}

POOL_STATUS ParameterStatus(POOL_CONNECTION *frontend, POOL_CONNECTION_POOL *backend)
{
	int len;
	int *len_array;
	int sendlen;
	char *p;
	char *name;
	char *value;

	pool_write(frontend, "S", 1);

	len_array = pool_read_message_length2(backend);

	if (len_array == NULL)
	{
		return POOL_END;
	}

	len = len_array[0];
	sendlen = htonl(len);
	pool_write(frontend, &sendlen, sizeof(sendlen));

	len -= 4;

	p = pool_read2(MASTER(backend), len);
	if (p == NULL)
		return POOL_END;

	name = p;
	value = p + strlen(name) + 1;

	pool_debug("name: %s value: %s", name, value);

	pool_add_param(&MASTER(backend)->params, name, value);

#ifdef DEBUG
	pool_param_debug_print(&MASTER(backend)->params);
#endif

	if (DUAL_MODE)
	{
		char *sp;

		if ((sp = pool_read2(SECONDARY(backend), len_array[1]-4)) == NULL)
			return POOL_END;

		name = sp;
		value = sp + strlen(name) + 1;

		pool_debug("secondary name: %s value: %s", name, value);
	}

	return pool_write(frontend, p, len);

}

/*
 * reset backend status. return values are:
 * 0: no query was issued 1: a query was issued 2: no more queries remain -1: error
 */
static int reset_backend(POOL_CONNECTION_POOL *backend, int qcnt)
{
	char *query;
	int qn;

	qn = pool_config.num_reset_queries;

	if (qcnt >= qn)
		return 2;

	query = pool_config.reset_query_list[qcnt];

	/* if transaction state is idle, we don't need to issue ABORT */
	if (TSTATE(backend) == 'I' && !strcmp("ABORT", query))
		return 0;

	if (Query(NULL, backend, query) != POOL_CONTINUE)
		return -1;

	return 1;
}

/*
 * return non 0 if load balance is possible
 */
static int load_balance_enabled(POOL_CONNECTION_POOL *backend, char *sql)
{
	if (pool_config.load_balance_mode &&
		DUAL_MODE &&
		MAJOR(backend) == PROTO_MAJOR_V3 &&
		TSTATE(backend) == 'I')
	{
		if (pool_config.ignore_leading_white_space)
		{
			/* ignore leading white spaces */
			while (*sql && isspace(*sql))
				sql++;
		}

		if (!strncasecmp(sql, "SELECT", 6))
			return 1;
	}
	return 0;
}

/*
 * start load balance mode
 */
static void start_load_balance(POOL_CONNECTION_POOL *backend)
{
	int i;
	int master;

	/* save backend connection slots */
	for (i=0;i<backend->num;i++)
	{
		slots[i] = backend->slots[i];
	}

	/* temporarily turn off replication mode */
	if (REPLICATION)
		replication_was_enabled = 1;
	if (MASTER_SLAVE)
		master_slave_was_enabled = 1;

	REPLICATION = 0;
	MASTER_SLAVE = 0;

	/* choose a master in random manner with weight */
	master = (random() <= weight_master)?0:1;
	backend->slots[0] = slots[master];
	pool_debug("start_load_balance: selected master is %d", master);

	/* start load balancing */
	in_load_balance = 1;
}

/*
 * finish load balance mode
 */
static void end_load_balance(POOL_CONNECTION_POOL *backend)
{
	int i;

	/* restore backend connection slots */
	for (i=0;i<backend->num;i++)
	{
		backend->slots[i] = slots[i];
	}

	/* turn on replication mode */
	REPLICATION = replication_was_enabled;
	MASTER_SLAVE = master_slave_was_enabled;

	replication_was_enabled = 0;
	master_slave_was_enabled = 0;
	in_load_balance = 0;

	pool_debug("end_load_balance: end load balance mode");
}

/*
 * send error message to frontend
 */
void pool_send_error_message(POOL_CONNECTION *frontend, int protoMajor,
							 char *code,
							 char *message,
							 char *detail,
							 char *hint,
							 char *file,
							 int line)
{
#define MAXDATA	1024
#define MAXMSGBUF 128
	if (protoMajor == PROTO_MAJOR_V2)
	{
		pool_write(frontend, "E", 1);
		pool_write_and_flush(frontend, message, strlen(message)+1);
	}
	else if (protoMajor == PROTO_MAJOR_V3)
	{
		char data[MAXDATA];
		char msgbuf[MAXMSGBUF];
		int len;
		int thislen;
		int sendlen;

		len = 0;

		pool_write(frontend, "E", 1);

		/* error level */
		thislen = snprintf(msgbuf, MAXMSGBUF, "SERROR");
		memcpy(data +len, msgbuf, thislen+1);
		len += thislen + 1;

		/* code */
		thislen = snprintf(msgbuf, MAXMSGBUF, "C%s", code);
		memcpy(data +len, msgbuf, thislen+1);
		len += thislen + 1;

		/* message */
		thislen = snprintf(msgbuf, MAXMSGBUF, "M%s", message);
		memcpy(data +len, msgbuf, thislen+1);
		len += thislen + 1;

		/* detail */
		if (*detail != '\0')
		{
			thislen = snprintf(msgbuf, MAXMSGBUF, "D%s", detail);
			memcpy(data +len, msgbuf, thislen+1);
			len += thislen + 1;
		}

		/* hint */
		if (*hint != '\0')
		{
			thislen = snprintf(msgbuf, MAXMSGBUF, "H%s", hint);
			memcpy(data +len, msgbuf, thislen+1);
			len += thislen + 1;
		}

		/* file */
		thislen = snprintf(msgbuf, MAXMSGBUF, "F%s", file);
		memcpy(data +len, msgbuf, thislen+1);
		len += thislen + 1;

		/* line */
		thislen = snprintf(msgbuf, MAXMSGBUF, "L%d", line);
		memcpy(data +len, msgbuf, thislen+1);
		len += thislen + 1;

		/* stop null */
		len++;
		*(data + len) = '\0';

		sendlen = len;
		len = htonl(len + 4);
		pool_write(frontend, &len, sizeof(len));
		pool_write_and_flush(frontend, data, sendlen);
	}
	else
		pool_error("send_error_message: unknown protocol major %d", protoMajor);
}

/*
 * sends q query in sync manner.
 * this function sends a query and wait for CommandComplete/ReadyForQuery.
 * if an error occured, it returns with POOL_ERROR.
 * this function does NOT handle SELECT/SHOW quries.
 * if no_ready_for_query is non 0, returns without reading the packet
 * length for ReadyForQuery. This mode is necessary when called from ReadyForQuery().
 */
static POOL_STATUS do_command(POOL_CONNECTION *backend, char *query, int protoMajor,
							  int no_ready_for_query)
{
	int len;
	int status;
	char kind;
	char *string;

	pool_debug("do_command: Query: %s", query);

	/* send the query to the backend */
	pool_write(backend, "Q", 1);
	len = strlen(query)+1;

	if (protoMajor == PROTO_MAJOR_V3)
	{
		int sendlen = htonl(len + 4);
		pool_write(backend, &sendlen, sizeof(sendlen));
	}

	if (pool_write_and_flush(backend, query, len) < 0)
	{
		return POOL_END;
	}

	/*
	 * Expecting CompleteCommand
	 */
	status = pool_read(backend, &kind, sizeof(kind));
	if (status < 0)
	{
		pool_error("do_command: error while reading message kind");
		return POOL_END;
	}

#ifdef NOT_USED
	if (kind != 'C')
	{
		pool_error("do_command: backend does not successfully complete command %s status %c", query, kind);

		/* the response must be an ERROR. handle it */
		
		return POOL_END;
	}
#endif
	/*
	 * read command tag of CommandComplete response
	 */
	if (protoMajor == PROTO_MAJOR_V3)
	{
		if (pool_read(backend, &len, sizeof(len)) < 0)
			return POOL_END;
		len = ntohl(len) - 4;
		string = pool_read2(backend, len);
		if (string == NULL)
			return POOL_END;
		pool_debug("command tag: %s", string);
	}
	else
	{
		string = pool_read_string(backend, &len, 0);
		if (string == NULL)
			return POOL_END;
	}

	/*
	 * Expecting ReadyForQuery
	 */
	status = pool_read(backend, &kind, sizeof(kind));
	if (status < 0)
	{
		pool_error("do_command: error while reading message kind");
		return POOL_END;
	}

	if (kind != 'Z')
	{
		pool_error("do_command: backend does not return ReadyForQuery");
		return POOL_END;
	}

	if (no_ready_for_query)
		return POOL_CONTINUE;

	if (protoMajor == PROTO_MAJOR_V3)
	{
		if (pool_read(backend, &len, sizeof(len)) < 0)
			return POOL_END;

		status = pool_read(backend, &kind, sizeof(kind));
		if (status < 0)
		{
			pool_error("do_command: error while reading transaction status");
			return POOL_END;
		}

		/* set transaction state */
		pool_debug("ReadyForQuery: transaction state: %c", kind);
		backend->tstate = kind;
	}

	return POOL_CONTINUE;
}

/*
 * judge if we need to lock the table
 * to keep SERIAL consistency among servers
 */
static int need_insert_lock(POOL_CONNECTION_POOL *backend, char *query)
{
	if (MAJOR(backend) != PROTO_MAJOR_V3)
		return 0;
	
	/*
	 * either insert_lock directive specified and without "NO INSERT LOCK" comment
	 * or "INSERT LOCK" comment exists?
	 */
	if ((pool_config.insert_lock && strncasecmp(query, NO_LOCK_COMMENT, NO_LOCK_COMMENT_SZ)) ||
		strncasecmp(query, LOCK_COMMENT, LOCK_COMMENT_SZ) == 0)
	{
		/* INSERT STATEMENT? */
		query = skip_comment(query);
		if (strncasecmp(query, "INSERT", 6) == 0)
			return 1;
	}

	return 0;
}

/*
 * if a transaction has not already started, start a new one.
 * issue LOCK TABLE IN SHARE ROW EXCLUSIVE MODE
 */
static POOL_STATUS insert_lock(POOL_CONNECTION_POOL *backend, char *query)
{
	char *table;
	char qbuf[1024];
	int i;

	if (MAJOR(backend) != PROTO_MAJOR_V3)
		return 0;

	/* if we are not in a transaction block,
	 * start a new transaction
	 */
	if (TSTATE(backend) == 'I')
	{
		for (i = 0;i < backend->num;i++)
		{
			if (do_command(backend->slots[i]->con, "BEGIN", PROTO_MAJOR_V3, 0) != POOL_CONTINUE)
				return POOL_END;
		}

		/* mark that we started new transaction */
		internal_transaction_started = 1;
	}

	/* issue lock table command */
	table = get_insert_command_table_name(query);
	snprintf(qbuf, sizeof(qbuf), "LOCK TABLE %s IN SHARE ROW EXCLUSIVE MODE", table);

	for (i = 0;i < backend->num;i++)
	{
		if (do_command(backend->slots[i]->con, qbuf, PROTO_MAJOR_V3, 0) != POOL_CONTINUE)
		{
			internal_transaction_started = 0;
			return POOL_END;
		}
	}

	return POOL_CONTINUE;
}

/*
 * obtain table name in INSERT statement
 */
static char *get_insert_command_table_name(char *query)
{
	static char table[1024];
	char *qbuf;
	char *token;

	table[0] = '\0';

	/* skip comment */
    query = skip_comment(query);

	if (*query == '\0')
		return table;

	/* skip spaces */
	while (*query && isspace(*query))
		query++;

	/* skip non spaces(INSERT) */
	while (*query && !isspace(*query))
		query++;

	/* skip spaces */
	while (*query && isspace(*query))
		query++;

	/* skip non spaces(INTO) */
	while (*query && !isspace(*query))
		query++;

	/* skip spaces */
	while (*query && isspace(*query))
		query++;

	/* get table */
	qbuf = strdup(query);
	token = strtok(qbuf, " (");
	strncpy(table, token, sizeof(table));
	free(qbuf);

	pool_debug("get_insert_command_table_name: extracted table name: %s", table);

	return table;
}

/* judge if this is a DROP DATABASE command */
static int is_drop_database(char *query)
{
	/* skip comment */
    query = skip_comment(query);

	if (*query == '\0')
		return 0;

	/* skip spaces */
	while (*query && isspace(*query))
		query++;

	/* DROP? */
	if (strncasecmp("DROP", query, 4))
		return 0;

	/* skip DROP */
	while (*query && !isspace(*query))
		query++;

	/* skip spaces */
	while (*query && isspace(*query))
		query++;

	/* DATABASE? */
	if (strncasecmp("DATABASE", query, 8))
		return 0;

	return 1;
}

/* skip SQL comments */
static char *skip_comment(char *query)
{
	if (strncmp(query, "/*", 2) == 0)
	{
		query += 2;
		while (query)
		{
			if (strncmp(query, "*/", 2) == 0)
			{
				query += 2;
				break;
			}
			query++;
		}
	}
	return query;
}
